import requests
import time
import json
import os

CONFIG_FILE = "config.txt"
LAST_MESSAGE_FILE = "last_message.txt"

def read_config():
    """Wczytuje ostatnio użyte IP ESP32 z pliku konfiguracyjnego."""
    if os.path.exists(CONFIG_FILE):
        with open(CONFIG_FILE, "r", encoding="utf-8") as f:
            lines = f.read().splitlines()
            if len(lines) >= 1:
                return lines[0]
    return ""

def save_config(esp32_ip):
    """Zapisuje IP ESP32 do pliku konfiguracyjnego."""
    with open(CONFIG_FILE, "w", encoding="utf-8") as f:
        f.write(esp32_ip + "\n")

def read_last_message():
    """Wczytuje ostatnią wysłaną wiadomość, aby uniknąć powtórzeń."""
    if os.path.exists(LAST_MESSAGE_FILE):
        with open(LAST_MESSAGE_FILE, "r", encoding="utf-8") as f:
            return f.read().strip()
    return ""

def save_last_message(msg):
    """Zapisuje ostatnią wysłaną wiadomość."""
    with open(LAST_MESSAGE_FILE, "w", encoding="utf-8") as f:
        f.write(msg)

def send_to_esp32(esp32_ip, message):
    """Wysyła wiadomość do ESP32 przez HTTP GET w formacie wymaganym przez ESP."""
    url = f"http://{esp32_ip}//&MSG={message}/&"
    try:
        requests.get(url, timeout=5)
        print(f"Wysłano: {message}")
    except requests.exceptions.ConnectionError as e:
        # Ignorujemy gwałtowne zamknięcie połączenia przez ESP
        if "10054" not in str(e):
            print(f"Błąd wysyłki: {e}")
    except requests.exceptions.RequestException as e:
        print(f"Błąd wysyłki: {e}")

def main():
    esp32_ip = read_config()
    if not esp32_ip:
        esp32_ip = input("Wpisz IP tekstscrollera: ").strip()
        if not esp32_ip:
            print("Nie podano IP ESP32!")
            return
        save_config(esp32_ip)

    last_message = read_last_message()

    server_url = input("Wpisz URL do pliku text.json (np. http://serwer:8000/led/text.json): ").strip()
    if not server_url:
        print("Nie podano URL!")
        return

    print("Rozpoczynam nasłuchiwanie nowej wiadomości... (Ctrl+C aby przerwać)")

    while True:
        try:
            r = requests.get(server_url, timeout=5)
            if r.status_code == 200:
                data = r.json()
                message = data.get("message", "").strip()
                if message and message != last_message:
                    send_to_esp32(esp32_ip, message)
                    last_message = message
                    save_last_message(message)
            time.sleep(3)  # odczekaj 3 sekundy przed kolejnym sprawdzeniem
        except requests.exceptions.RequestException:
            time.sleep(3)  # w przypadku błędu też odczekaj 3 sekundy

if __name__ == "__main__":
    main()
